class BookingWidget
{
    /**
     * Creates a new instance with the specified options.
     */
    constructor(options = {})
    {
        const defaultOptions = {

            cdn: '',

            // Widget type configuration.

            type: 'daterange',

            // Date restrictions.

            minDaysRange: null,
            maxDaysRange: null,

            // Parameters for built-in callback functions.

            mozLive3Parameters: {
                componentID: 0,
                componentName: '',
                componentSuperglobal: 0,
                serviceID: 0,
                onWidgetInitializedAction: '',
                onDateFromSelectedAction: '',
                onTimeFromSelectedActions: '',
                onDateToSelectedAction: '',
            },

            // Callback functions.

            onSubmit: null,

            // Localization strings.

            localization: {
                txtSelect: 'Select',
                txtSelectTime: 'Select time',
                txtCancel: 'Cancel',
                txtCalendar: 'Calendar',
                txtNotSelected: 'Not selected',
                txtFrom: 'From',
                txtTo: 'To',
                txtTitle: 'Select date and / or time',
                txtPreviousMonth: 'Previous month',
                txtNextMonth: 'Next month',
                txtMonthNames: null,
                txtDayNames: null
            }
        };

        this.options = $.extend(true, {}, defaultOptions, options);

        this.widget = null;

        this.bookingCalendarFrom = null;
        this.bookingCalendarTo = null;

        this.bookingTimes = null;
        this.bookingTimesFrom = null;
        this.bookingTimesTo = null;
    }

    /**
     * Dynamically loads a CSS stylesheet into the document.
     */
    #loadCSS(name, url)
    {
        return new Promise((resolve) => {

            if ($(`link.${name}`).length) {
                resolve();
                return;
            }

            const link = $('<link/>', {
                rel: 'stylesheet',
                type: 'text/css',
                class: name,
                href: url
            });

            link.on('load', resolve);
            link.appendTo('head');
        });
    }

    /**
     * Dynamically loads a JavaScript file into the document by appending a script element to the head.
     */
    #loadJS(name, url)
    {
        return new Promise((resolve) => {

            if (document.querySelector(`script.${name}`)) {
                resolve();
                return;
            }

            const script = document.createElement('script');
            script.src = url;
            script.onload = () => {
                script.classList.add(name);
                resolve();
            }
            document.head.appendChild(script);
        });
    }

    /**
     * Initializes the widget template and appends it to the DOM.
     */
    #initWidget()
    {
        // Initializes the Widget template.

        this.widget = $(`
            <form class="booking-panel moze-form ${this.options.type}">
                <div class="booking-panel-main">
                    <div class="booking-panel-top">
                        <button id="booking-panel-button-from" type="button" class="from clearbutton selected">
                            <div>${this.options.localization.txtFrom}</div>
                            <div class="date">${this.options.localization.txtNotSelected}</div>
                            <div class="time">&nbsp;</div>
                        </button>
                        <button id="booking-panel-button-to" type="button" class="to clearbutton">
                            <div>${this.options.localization.txtTo}</div>
                            <div class="date">${this.options.localization.txtNotSelected}</div>
                            <div class="time">&nbsp;</div>
                        </button>
                    </div>
                    <div class="booking-panel-middle">
                        <div class="loading-screen" style="display: none"></div>
                        <div class="booking-panel-side from">
                            <div id="booking-calendar"></div>
                            <select class="ctrl booking-times from" size="5" disabled aria-label="${this.options.localization.txtSelectTime}"></select>
                        </div>
                        <div class="booking-panel-side to hidden">
                            <div id="booking-calendar-to"></div>
                            <select class="ctrl booking-times to" size="5" disabled aria-label="${this.options.localization.txtSelectTime}"></select>
                        </div>
                    </div>
                </div>
                <div class="booking-panel-bottom mz_inlinebuttons">
                    <button type="submit" class="moze-button-large save" disabled>${this.options.localization.txtSelect}</button>
                    <button type="button" class="moze-button-large cancel btn-alt">${this.options.localization.txtCancel}</button>
                </div>
            </form>
        `);

        $('body').prepend(this.widget);

        this.bookingTimes = this.widget.find('select.booking-times');
        this.bookingTimesFrom = this.widget.find('select.booking-times.from');
        this.bookingTimesTo = this.widget.find('select.booking-times.to');

        this.#initCalendars();
        this.#initEvents();
    }

    /**
     * Initializes one or two MinimalistCalendar instances based on the configuration options.
     */
    #initCalendars()
    {
        // Initializes the Minimalist calendar.

        this.bookingCalendarFrom = new MinimalistCalendar({

            containerId: 'booking-calendar',
            maxPastMonths: 0,
            maxFutureMonths: 12,
            disablePastDays: true,
            disabledDays: [],
            enableRangeSelection: (this.options.type == 'daterange'),
            monthNames: this.options.localization.txtMonthNames,
            dayNames: this.options.localization.txtDayNames,
            txtPreviousMonth: this.options.localization.txtPreviousMonth,
            txtNextMonth: this.options.localization.txtNextMonth,
            minDaysRange: this.options.minDaysRange,
            maxDaysRange: this.options.maxDaysRange,

            onDateSelected: (selectedDates) => {

                // Disables dates and times in the "to" calendar.

                if (this.options.type == 'timerange') {
                    this.bookingCalendarTo.disableAllDays();
                    this.bookingCalendarTo.renderCalendar();
                    this.#updateAvailableTimes(null, true);
                }

                // Triggers an event that should update calendars based on the selected date.

                this.#beginLoading();
                this.#onDateFromSelected(selectedDates[0])
                    .then((response) => {
                        if (!response) {
                            return;
                        }
                        this.#updateAvailableTimes(response.timeFrom);
                    })
                    .always(() => {
                        this.#updateSelection();
                        this.#endLoading();
                    });
            }
        });

        // Initializes the second Minimalist calendar for a complex mode view.

        if (this.options.type == 'timerange') {

            this.bookingCalendarTo = new MinimalistCalendar({

                containerId: 'booking-calendar-to',
                maxPastMonths: 0,
                maxFutureMonths: 12,
                disablePastDays: true,
                disabledDays: [],
                enableRangeSelection: false,
                monthNames: this.options.localization.txtMonthNames,
                dayNames: this.options.localization.txtDayNames,
                txtPreviousMonth: this.options.localization.txtPreviousMonth,
                txtNextMonth: this.options.localization.txtNextMonth,

                onDateSelected: (selectedDates) => {

                    // Triggers an event that should update calendars based on the selected date.

                    this.#beginLoading();
                    this.#onDateToSelected(this.bookingCalendarFrom.selectedDays[0], this.bookingTimesFrom.val(), selectedDates[0])
                        .then((response) => {
                            if (!response) {
                                return;
                            }
                            this.#updateAvailableTimes(response.timeTo, true);
                        })
                        .always(() =>{
                            this.#updateSelection();
                            this.#endLoading();
                        });
                }
            });
        }
    }

    /**
     * Initializes event handlers for the widget.
     */
    #initEvents()
    {
        if (this.options.type == 'timerange') {

            // Toggle between From/To panels, show the selected panel and focus the first button.

            this.widget.find('.booking-panel-top button').click((e) => {
                $('.booking-panel-top button').removeClass('selected');
                $('.booking-panel-side').addClass('hidden');
                $(e.currentTarget ).addClass('selected');
                let selectedPanel;
                if ($(e.currentTarget ).hasClass('to')) {
                    selectedPanel = $('.booking-panel-side.to');
                }
                else {
                    selectedPanel = $('.booking-panel-side.from');
                }
                let focusable = selectedPanel.removeClass('hidden').find('button:visible:not(:disabled)').first();
                focusable.focus();
            });

            // Sync top button selection when the side panel is activated via keyboard/click.

            this.widget.find('.booking-panel-side').on('keypress click', (e) => {
                let index = $(e.currentTarget).parent().children('.booking-panel-side').index(e.currentTarget);
                $('.booking-panel-top button').removeClass('selected');
                $('.booking-panel-top button').eq(index).addClass('selected');
            });

            // Set ARIA labels for calendar accessibility

            this.widget.find('#booking-calendar').attr('aria-labelledby', 'booking-panel-button-from');
            this.widget.find('#booking-calendar-to').attr('aria-labelledby', 'booking-panel-button-to');
        }
        else {
            this.widget.find('#booking-calendar').attr('aria-label', this.options.localization.txtCalendar);
        }

        // More events and initializations.

        $(window).on('resize.booking-panel', () => {
            this.#resizeWidget();
        });
        this.#resizeWidget();

        this.widget.find('.booking-panel-top').toggle(this.options.type == 'timerange');
        this.widget.find('.booking-panel-side select.booking-times').toggle(this.options.type == 'timerange' || this.options.type == 'time');

        this.bookingTimesFrom.change(() => {
            let selectedDateFrom = this.bookingCalendarFrom.selectedDays[0],
                selectedTimeFrom = this.bookingTimesFrom.val();
            this.#beginLoading();
            this.#onTimeFromSelected(selectedDateFrom, selectedTimeFrom)
                .then((response) => {
                    if (this.bookingCalendarTo) {
                        if (!response) {
                            this.bookingCalendarTo.disableAllDays();
                        }
                        else {
                            this.bookingCalendarTo.selectedDays = [];
                            this.bookingCalendarTo.disableDaysBefore = selectedDateFrom;
                            this.bookingCalendarTo.disabledDays = response.disabledDatesTo;
                        }
                        this.bookingCalendarTo.renderCalendar();
                    }
                })
                .always(() => {
                    this.#updateSelection();
                    this.#endLoading();
                });
        });

        this.bookingTimesTo.change(() => {
            // No need to do anything for now.
        });

        this.widget.find('button.cancel').click(() => {
            this.#closeWidget();
        });

        // Submit procedure.

        this.widget.on('submit', (event) => {
            event.preventDefault();
            if (typeof this.options.onSubmit === 'function') {
                $(event.currentTarget).addClass('animate-loading');
                let resultData = this.#getDataFromWidget();
                this.#beginLoading();
                this.options
                    .onSubmit(this, resultData)
                    .then((result) => {
                        if (result) {
                            this.#closeWidget();
                        }
                    })
                    .always(() => {
                        this.#endLoading();
                    });
            }
            else {
                this.#closeWidget();
            }
        });
    }

    /**
     * Updates the current selection of dates and times based on the user's input.
     */
    #updateSelection()
    {
        let dateFrom, timeFrom, dateTo, timeTo;
        let enableSubmit = false;

        if ((this.bookingCalendarFrom.selectedDays.length >= (this.options.minDaysRange ?? 1)) &&
            (this.options.maxDaysRange === null || this.bookingCalendarFrom.selectedDays.length <= this.options.maxDaysRange)) {

            enableSubmit = true;
            dateFrom = this.bookingCalendarFrom.selectedDays[0];
            if (this.options.type == 'time' || this.options.type == 'timerange') {
                timeFrom = this.bookingTimesFrom.val();
                enableSubmit = timeFrom !== null && timeFrom !== '';
            }
        }

        if (this.options.type == 'timerange') {

            if (this.bookingCalendarTo.selectedDays.length > 0) {
                dateTo = this.bookingCalendarTo.selectedDays[0];
                timeTo = this.bookingTimesTo.val();
                enableSubmit = enableSubmit && (timeTo !== null && timeTo !== '');
            }
            else {
                enableSubmit = false;
            }
        }

        timeFrom = timeFrom ? timeFrom : '&nbsp;';
        dateFrom = dateFrom ? dateFrom : this.options.localization.txtNotSelected;
        timeTo = timeTo ? timeTo : '&nbsp;';
        dateTo = dateTo ? dateTo : this.options.localization.txtNotSelected;

        this.widget.find('button.from .date').html(dateFrom);
        this.widget.find('button.from .time').html(timeFrom);
        this.widget.find('button.to .date').html(dateTo);
        this.widget.find('button.to .time').html(timeTo);
        this.widget.find('button.save').prop('disabled', !enableSubmit);
    }

    /**
     * Updates the available times in the dropdown selection for booking.
     */
    #updateAvailableTimes(availableTimes, isTo = false, triggerChange = true)
    {
        let selectCtrl = isTo ? this.bookingTimesTo : this.bookingTimesFrom;

        let savedValue = selectCtrl.val();
        selectCtrl.empty();

        if (Array.isArray(availableTimes)) {
            $.each(availableTimes, (index, time) => {
                let $option = $('<option></option>')
                    .attr('value', time.value)
                    .text(time.text)
                    .prop('disabled', time.disabled);
                selectCtrl.append($option);
            });
            if (savedValue && selectCtrl.find(`option[value="${savedValue}"]`).length > 0) {
                selectCtrl.val(savedValue);
            }
            else {
                selectCtrl.find('option:not(:disabled)').first().prop('selected', true);
            }
        }

        selectCtrl.prop('disabled', !(Array.isArray(availableTimes) && availableTimes.length));

        if (triggerChange) {
            selectCtrl.trigger('change');
        }
    }

    /**
     * Initiates the loading state for the widget by displaying the loading screen.
     */
    #beginLoading()
    {
        this.widget.find('.loading-screen').show();
        this.widget.find('button[type=submit]').prop('disabled', true);
    }

    /**
     * Hides the loading state.
     */
    #endLoading()
    {
        this.widget.find('.loading-screen').hide();
        this.#updateSelection();
    }

    /**
     * Adjusts the widget's size dynamically based on the type option and the window's dimensions.
     */
    #resizeWidget()
    {
        if (this.options.type == 'timerange' || window.innerWidth <= 750 || window.innerHeight <= 600) {
            this.bookingTimes.removeAttr('size');
        }
        else {
            this.bookingTimes.attr('size', 5);
        }
    }

    /**
     * Loads the required CSS and JavaScript dependencies for the calendar widget, then
     * displays the widget within a modal popup.
     */
    #showWidget(stateData)
    {
        const cssPromise = this.#loadCSS('calendar-css', this.options.cdn + '/libs/js/calendar/calendar.css?1');
        const jsPromise = this.#loadJS('calendar-js', this.options.cdn + '/libs/js/calendar/calendar.js?1');

        return Promise.all([cssPromise, jsPromise])
            .then(() => {

                this.#initWidget();

                 simpleModalPopup.show({
                    content: this.widget,
                    txtTitle: this.options.localization.txtTitle,
                    onShow: () => {

                        if (stateData) {
                            this.restoreState(stateData);
                        }
                        else {
                            this.bookingCalendarFrom.disableAllDays();
                            if (this.bookingCalendarTo) {
                                this.bookingCalendarTo.disableAllDays();
                            }
                            this.#onWidgetInitialized();
                            this.#updateSelection();
                        }

                        this.widget.find('.booking-panel-side button:not(:disabled)').first().focus();
                    }
                });
            });
    }

    /**
     * Closes the booking widget by closing the modal popup.
     */
    #closeWidget()
    {
        $(window).off('resize.booking-panel');
        simpleModalPopup.close();
    }

    /**
     * Retrieves data from the widget based on the selected values.
     */
    #getDataFromWidget()
    {
        let result = {
            dateFrom: null,
            timeFrom: null,
            dateTo: null,
            timeTo: null,
            formatted: ''
        };

        switch (this.options.type) {

            case 'date':
                result.dateFrom = this.bookingCalendarFrom.selectedDays[0];
                break;

            case 'daterange':
                result.dateFrom = this.bookingCalendarFrom.selectedDays[0];
                result.dateTo = this.bookingCalendarFrom.selectedDays.at(-1);
                break;

            case 'time':
                result.dateFrom = this.bookingCalendarFrom.selectedDays[0];
                result.timeFrom = this.bookingTimesFrom.val();
                break;

            case 'timerange':
                result.dateFrom = this.bookingCalendarFrom.selectedDays[0];
                result.timeFrom = this.bookingTimesFrom.val();
                result.dateTo = this.bookingCalendarTo.selectedDays[0];
                result.timeTo = this.bookingTimesTo.val();
                break;
        }

        if (result.dateFrom != null) {
            result.formatted = this.#formatResult(result);
        }

        return result;
    }

    /**
     * Formats the given result object into a date/time string.
     */
    #formatResult(result)
    {
        const formatDate = (date) => {
            if (date == null) {
                return '';
            }
            const formatter = new Intl.DateTimeFormat('en-CA', {
                year: 'numeric',
                month: '2-digit',
                day: '2-digit'
            });
            return formatter.format(new Date(date));
        };

        const formatTime = (date, time) => {
            if (date == null || time == null) {
                return '';
            }
            const formatter = new Intl.DateTimeFormat('en-GB', {
                hour: '2-digit',
                minute: '2-digit'
            });
            return formatter.format(new Date(`${date} ${time}`));
        };

        const
            fDateFrom = formatDate(result.dateFrom),
            fTimeFrom = formatTime(result.dateFrom, result.timeFrom),
            fDateTo = formatDate(result.dateTo),
            fTimeTo = formatTime(result.dateTo, result.timeTo);

        let formatted = '';

        switch (this.options.type) {

            case 'date':
                formatted = fDateFrom;
                break;

            case 'daterange':
                formatted = `${fDateFrom} - ${fDateTo}`;
                break;

            case 'time':
                formatted = `${fDateFrom} ${fTimeFrom}`;
                break;

            case 'timerange':
                formatted = `${fDateFrom} ${fTimeFrom} - ${fDateTo} ${fTimeTo}`;
                break;
        }

        return formatted;
    }

    /**
     * Converts a date string in the format 'YYYY-MM-DD' to a Date object.
     * Returns the Date object or null if the input string represents an invalid date.
     */
    #stringToDate(dateString)
    {
        if (!/^\d{4}-\d{2}-\d{2}$/.test(dateString)) {
            return null;
        }

        const [year, month, day] = dateString.split('-').map(Number);

        if (month < 1 || month > 12 || day < 1 || day > 31) {
            return null;
        }

        const date = new Date(Date.UTC(year, month - 1, day));

        if (date.getFullYear() !== year ||
            date.getMonth() !== month - 1 ||
            date.getDate() !== day) {
            return null;
        }

        return date;
    }

    /**
     * Returns the data that describes the widget's current state.
     */
    saveState()
    {
        const stateData = {

            type: this.options.type,

            disabledDatesFrom: this.bookingCalendarFrom.disabledDays,
            disabledDatesTo: this.bookingCalendarTo ? this.bookingCalendarTo.disabledDays : [],

            timesFrom: this.bookingTimesFrom.find('option').map((index, option) => {
                return {
                    value: $(option).val(),
                    text: $(option).text()
                };
            }).get(),

            timesTo: this.bookingTimesTo.find('option').map((index, option) => {
                return {
                    value: $(option).val(),
                    text: $(option).text()
                };
            }).get(),

            selection: this.#getDataFromWidget()
        };

        return stateData;
    }

    /**
     * Loads the widget state from the specified data.
     */
    restoreState(stateData)
    {
        const defaultStateData = {

            type: '',

            disabledDatesFrom: [],
            disabledDatesTo: [],
            timesFrom: [],
            timesTo: [],

            selection: {
                dateFrom: null,
                timeFrom: null,
                dateTo: null,
                timeTo: null
            }
        };

        stateData = $.extend(true, {}, defaultStateData, stateData);

        // Functions to restore a specific part of states.

        const restoreCalendar = (calendarCtrl, disabledDates, selectedDateString) => {

            if (calendarCtrl == null) {
                return;
            }

            calendarCtrl.disableAllDays();

            const selectedDate = this.#stringToDate(selectedDateString);
            if (selectedDate == null) {
                return;
            }

            calendarCtrl.disabledDays = disabledDates;
            calendarCtrl.currentMonth = selectedDate.getMonth();
            calendarCtrl.currentYear = selectedDate.getFullYear();
            calendarCtrl.selectedDays = [selectedDate.toISOString().split('T')[0]];
            calendarCtrl.renderCalendar();
        };

        const restoreTimes = (timeCtrl, isToSelectCtrl, times, selectedTimeString) => {

            this.#updateAvailableTimes([], isToSelectCtrl, false);

            if (!$.isArray(stateData.timesFrom)) {
                return;
            }

            this.#updateAvailableTimes(times, isToSelectCtrl, false);
            timeCtrl.val(selectedTimeString);
        };

        // Restores the state of the widget.

        if (this.options.type == 'date') {
            restoreCalendar(this.bookingCalendarFrom, stateData.disabledDatesFrom, stateData.selection.dateFrom);
        }

        if (this.options.type == 'time') {
            restoreCalendar(this.bookingCalendarFrom, stateData.disabledDatesFrom, stateData.selection.dateFrom);
            restoreTimes(this.bookingTimesFrom, false, stateData.timesFrom, stateData.selection.timeFrom);
        }

        if (this.options.type == 'daterange') {
            restoreCalendar(this.bookingCalendarFrom, stateData.disabledDatesFrom, stateData.selection.dateFrom);
            restoreCalendar(this.bookingCalendarTo, stateData.disabledDatesTo, stateData.selection.dateTo);
        }

        if (this.options.type == 'timerange') {
            restoreCalendar(this.bookingCalendarFrom, stateData.disabledDatesFrom, stateData.selection.dateFrom);
            restoreTimes(this.bookingTimesFrom, false, stateData.timesFrom, stateData.selection.timeFrom);
            restoreCalendar(this.bookingCalendarTo, stateData.disabledDatesTo, stateData.selection.dateTo);
            restoreTimes(this.bookingTimesTo, true, stateData.timesTo, stateData.selection.timeTo);
        }

        this.#updateSelection();
    }

    /**
     * Builds the source parameter for MozLive requests.
     */
    #buildMozLiveSource()
    {
        if (this.options.mozLive3Parameters.componentSuperglobal == 1) {
            return {
                name: this.options.mozLive3Parameters.componentName,
                superglobal: 1
            }
        }
        else {
            return {
                id: this.options.mozLive3Parameters.componentID,
                name: this.options.mozLive3Parameters.componentName
            }
        }
    }

    /**
     * Handles the final initialization stage of the widget.
     */
    #onWidgetInitialized()
    {
        // Request available from dates.

        this.#beginLoading();

        new mozLive3({
            source: this.#buildMozLiveSource(),
            action: this.options.mozLive3Parameters.onWidgetInitializedAction,
            parameters: {
                service: this.options.mozLive3Parameters.serviceID
            },
            responseOnError: true,
            response: {
                callback: [
                    (response) => {
                        if (response.error) {
                            return;
                        }
                        if (response.data) {
                            this.bookingCalendarFrom.disabledDays = response.data.disabledDatesFrom;
                        }
                        this.bookingCalendarFrom.renderCalendar();
                        this.#updateSelection();
                        this.#endLoading();
                    }
                ]
            }
        });
    }

    /**
     * Handles the selection of a date in the "from" calendar.
     */
    #onDateFromSelected(selectedDate)
    {
        let deferred = $.Deferred();

        if (typeof selectedDate === 'undefined') {

            // A date was unselected.

            if (this.bookingCalendarTo) {
                this.bookingCalendarTo.selectedDays = [];
                this.bookingCalendarTo.disableAllDays();
            }

            this.#updateAvailableTimes(null, false);
            this.#updateAvailableTimes(null, true);

            return deferred.promise();
        }
        else {

            // A date was selected.

            if (this.bookingCalendarTo) {
                const selectedDateObj = this.#stringToDate(selectedDate);
                if (selectedDateObj) {
                    this.bookingCalendarTo.currentMonth = selectedDateObj.getMonth();
                    this.bookingCalendarTo.currentYear = selectedDateObj.getFullYear();
                    this.bookingCalendarTo.renderCalendar();
                }
            }
        }

        // No need to select the time in date-only calendars.

        if (this.options.type == 'date' || this.options.type == 'daterange') {
            deferred.resolve(null);
            return deferred.promise();
        }

        // Request available from times.

        new mozLive3({
            source: this.#buildMozLiveSource(),
            action: this.options.mozLive3Parameters.onDateFromSelectedAction,
            parameters: {
                service: this.options.mozLive3Parameters.serviceID,
                date: selectedDate
            },
            responseOnError: true,
            response: {
                callback: [
                    (response) => {
                        if (response.error) {
                            deferred.resolve(null);
                        }
                        else {
                            deferred.resolve(response.data);
                        }
                    }
                ]
            }
        });

        // Ready!

        return deferred.promise();
    }

    /**
     * Handles the selection of a time in the "from" calendar.
     */
    #onTimeFromSelected(selectedDate, selectedTime)
    {
        let deferred = $.Deferred();

        // No need to select anything for a simple date/time calendar without an additional secondary calendar.

        if (this.options.type != 'timerange') {
            deferred.resolve(null);
            return deferred.promise();
        }

        // Request available "to" dates.

        new mozLive3({
            source: this.#buildMozLiveSource(),
            action: this.options.mozLive3Parameters.onTimeFromSelectedActions,
            parameters: {
                service: this.options.mozLive3Parameters.serviceID,
                date: selectedDate,
                time: selectedTime
            },
            responseOnError: true,
            response: {
                callback: [
                    (response) => {
                        if (response.error) {
                            deferred.resolve(null);
                        }
                        else {
                            deferred.resolve(response.data);
                        }
                    }
                ]
            }
        });

        // Ready!

        return deferred.promise();
    }

    /**
     * Handles the selection of a date in the "to" calendar.
     */
    #onDateToSelected(selectedDateFrom, selectedTimeFrom, selectedDateTo)
    {
        let deferred = $.Deferred();

        // No need to select anything for a simple date/time calendar without an additional secondary calendar.

        if (this.options.type != 'timerange') {
            deferred.resolve(null);
            return deferred.promise();
        }

        if (typeof selectedDateTo === 'undefined') {
            this.#updateAvailableTimes(null, true);
        }

        // Request available "to" times.

        new mozLive3({
            source: this.#buildMozLiveSource(),
            action: this.options.mozLive3Parameters.onDateToSelectedAction,
            parameters: {
                service: this.options.mozLive3Parameters.serviceID,
                dateFrom: selectedDateFrom,
                timeFrom: selectedTimeFrom,
                dateTo: selectedDateTo
            },
            responseOnError: true,
            response: {
                callback: [
                    (response) => {
                        if (response.error) {
                            deferred.resolve(null);
                        }
                        else {
                            deferred.resolve(response.data);
                        }
                    }
                ]
            }
        });

        // Ready!

        return deferred.promise();
    }

    /**
     * Executes the show method, loading necessary dependencies asynchronously.
     */
    async show(stateData)
    {
        try {
            await this.#showWidget(stateData);
        }
        catch (error) {
            console.error(error);
        }
    }
}